#!/usr/bin/env python3
#
# This script performs initial setup of the HMI with: hmi_sn, EQ-#, and sn.
#       - Creates /opt/OGic-2/run/config/systems.jon
#       - Adds Mac Address Reboot entry in crontab. 
#
# NOTE: Is not recommanded to change crontab this way because this may corrupt the file.
#       But seems to work. Make sure to look at the printout after running command.
#
# SYNOPSIS and usage example (must run with sudo):
# sudo python3 hmi_init_setup.py hmi_sn EQ-Proj  sn
#                                ------ -------- -----------
# sudo python3 hmi_init_setup.py 240001 EQ-26612 62024031031

# NOTE: Is ok to run this command again if entered wrong number before.


import json
import os
import sys
import subprocess


def generateSystemFile(proj, sn):
    # Define the data structure for JSON content
    data = {
        "controller": {
            "projectNum_SYS": proj,
            "serialNum_SYS": sn
        }
    }

    # Specify the directory and filename
    directory = "/opt/OGic-2/run/config/"
    filename = "systems.json"
    filepath = os.path.join(directory, filename)

    # Ensure the directory exists
    os.makedirs(directory, exist_ok=True)

    # Write the JSON content to the file
    with open(filepath, 'w') as json_file:
        json.dump(data, json_file, indent=2)

    print(f"JSON file has been created at {filepath}")


def formatMacAddress(hmi_sn):
    # Split the hmi_sn into groups of two digits
    groups = [hmi_sn[i:i+2] for i in range(0, len(hmi_sn), 2)]
    
    # Convert each group to hexadecimal and format with leading zero if necessary
    hex_groups = [f"{int(group):02x}" for group in groups]
    
    # Assemble the MAC address
    mac_address = "c2:56:30:" + ":".join(hex_groups)
    return mac_address


def setMacAddr(hmi_sn):
    # Define the base command pattern to look for in existing crontab entries
    base_command_pattern = "@reboot /usr/sbin/ip link set dev eth0 address"
    # Define the full new cron command with the MAC address
    cron_command = f"{base_command_pattern} {formatMacAddress(hmi_sn)}"
    
    try:
        # List current crontab; if none exists, handle gracefully
        current_crontab_result = subprocess.run(['crontab', '-l'], capture_output=True, text=True)
        current_crontab_content = current_crontab_result.stdout if current_crontab_result.returncode == 0 else ""
        
        # Filter out any existing lines that start with the base command pattern
        updated_crontab_lines = [line for line in current_crontab_content.splitlines() 
                                 if not line.startswith(base_command_pattern)]

        # Add the new cron command to the list of crontab lines
        updated_crontab_lines.append(cron_command)
        
        # Join the updated list back into a single string to form the new crontab content
        updated_crontab_content = "\n".join(updated_crontab_lines) + "\n"
        
        # Update the user's crontab with the new content
        subprocess.run(['crontab', '-'], input=updated_crontab_content, text=True, check=True)
        print(f"Cron job updated to set MAC address to {formatMacAddress(hmi_sn)}.")
        
    except subprocess.CalledProcessError as e:
        print(f"Error updating crontab: {e}")


def showCrontabContent():
    # Not recommanded to change crontab this way because this may corrupt the file
    # but seems to work. Make sure to look at the printout after running command.
    try:
        # Attempt to show current crontab content; handle case where no crontab exists
        result = subprocess.run(['crontab', '-l'], capture_output=True, text=True)
        if result.returncode == 0:
            print("Current crontab content:")
            print(result.stdout)
        else:
            print("No crontab for this user.")
    except subprocess.CalledProcessError as e:
        print("Error showing crontab content:", e)


def main():
    # Check if three arguments are passed (excluding the script name)
    if len(sys.argv) != 4:
        print("Usage: python3 script.py <proj> <sn> <hmi_sn>")
        sys.exit(1)
    hmi_sn = sys.argv[1]  # Capture the third input argument for HMI serial number
    proj = sys.argv[2]
    sn = sys.argv[3]


    generateSystemFile(proj, sn)
    setMacAddr(hmi_sn)  # Pass the HMI serial number to setMacAddr
    showCrontabContent()  # Display the current crontab content


if __name__ == "__main__":
    main()

